//
//  TrendChartView.swift
//  death_app Watch App
//
//  Created by Task-Master AI
//

import SwiftUI
import Charts

struct TrendChartView: View {
    let historicalData: [PredictionHistoryEntry]
    let timeframe: HistoricalTimeframe
    
    @State private var selectedDataPoint: PredictionHistoryEntry?
    
    var body: some View {
        VStack(alignment: .leading, spacing: 8) {
            chartHeaderView
            
            Chart(historicalData) { entry in
                LineMark(
                    x: .value("Time", entry.timestamp),
                    y: .value("Life Expectancy", entry.lifeExpectancy)
                )
                .foregroundStyle(.blue)
                .lineStyle(StrokeStyle(lineWidth: 2))
                
                AreaMark(
                    x: .value("Time", entry.timestamp),
                    y: .value("Life Expectancy", entry.lifeExpectancy)
                )
                .foregroundStyle(
                    LinearGradient(
                        colors: [.blue.opacity(0.3), .clear],
                        startPoint: .top,
                        endPoint: .bottom
                    )
                )
                
                if let selectedDataPoint = selectedDataPoint,
                   selectedDataPoint.id == entry.id {
                    PointMark(
                        x: .value("Time", entry.timestamp),
                        y: .value("Life Expectancy", entry.lifeExpectancy)
                    )
                    .foregroundStyle(.blue)
                    .symbolSize(50)
                }
            }
            .frame(height: 120)
            .chartBackground { chartProxy in
                GeometryReader { geometry in
                    Rectangle()
                        .fill(Color.clear)
                        .contentShape(Rectangle())
                        .gesture(
                            DragGesture(minimumDistance: 0)
                                .onChanged { value in
                                    updateSelectedDataPoint(
                                        at: value.location,
                                        geometry: geometry,
                                        chartProxy: chartProxy
                                    )
                                }
                                .onEnded { _ in
                                    selectedDataPoint = nil
                                }
                        )
                }
            }
            .chartXAxis {
                AxisMarks(values: .stride(by: xAxisStride)) { value in
                    AxisTick()
                    AxisValueLabel {
                        if let date = value.as(Date.self) {
                            Text(formatXAxisLabel(date: date))
                                .font(.caption2)
                        }
                    }
                }
            }
            .chartYAxis {
                AxisMarks { value in
                    AxisTick()
                    AxisValueLabel {
                        if let years = value.as(Double.self) {
                            Text("\(Int(years))y")
                                .font(.caption2)
                        }
                    }
                }
            }
            
            if let selectedDataPoint = selectedDataPoint {
                dataPointDetailView(for: selectedDataPoint)
            }
        }
        .padding(.horizontal, 4)
    }
    
    private var chartHeaderView: some View {
        HStack {
            VStack(alignment: .leading, spacing: 2) {
                Text("Life Expectancy Trend")
                    .font(.caption)
                    .foregroundColor(.secondary)
                
                if let latest = historicalData.last {
                    Text("\(formatLifeExpectancy(latest.lifeExpectancy))")
                        .font(.title3.bold())
                        .foregroundColor(.primary)
                }
            }
            
            Spacer()
            
            trendIndicator
        }
    }
    
    private var trendIndicator: some View {
        HStack(spacing: 4) {
            if historicalData.count >= 2 {
                let first = historicalData.first!.lifeExpectancy
                let last = historicalData.last!.lifeExpectancy
                let change = last - first
                
                Image(systemName: change > 0 ? "arrow.up.right" : change < 0 ? "arrow.down.right" : "arrow.right")
                    .font(.caption)
                    .foregroundColor(change > 0 ? .green : change < 0 ? .red : .orange)
                
                Text(formatChange(change))
                    .font(.caption.bold())
                    .foregroundColor(change > 0 ? .green : change < 0 ? .red : .orange)
            }
        }
    }
    
    private func dataPointDetailView(for dataPoint: PredictionHistoryEntry) -> some View {
        VStack(alignment: .leading, spacing: 4) {
            HStack {
                Text("Selected Point")
                    .font(.caption.bold())
                Spacer()
                Text(formatDate(dataPoint.timestamp))
                    .font(.caption)
                    .foregroundColor(.secondary)
            }
            
            HStack {
                Text("Life Expectancy:")
                    .font(.caption)
                Spacer()
                Text(formatLifeExpectancy(dataPoint.lifeExpectancy))
                    .font(.caption.bold())
            }
            
            if let heartRate = dataPoint.avgHeartRate {
                HStack {
                    Text("Heart Rate:")
                        .font(.caption)
                    Spacer()
                    Text("\(Int(heartRate)) bpm")
                        .font(.caption)
                }
            }
        }
        .padding(8)
        .background(Color.gray)
        .cornerRadius(8)
        .transition(.opacity)
    }
    
    private var xAxisStride: Calendar.Component {
        switch timeframe {
        case .day: return .hour
        case .week: return .day
        case .month: return .weekOfYear
        case .year: return .month
        }
    }
    
    private func updateSelectedDataPoint(
        at location: CGPoint,
        geometry: GeometryProxy,
        chartProxy: ChartProxy
    ) {
        let plotFrame = chartProxy.plotAreaFrame
        
        let xPosition = location.x - geometry[plotFrame].minX
        let xValue = chartProxy.value(atX: xPosition, as: Date.self) ?? Date()
        
        // Find the closest data point
        let closestEntry = historicalData.min { entry1, entry2 in
            abs(entry1.timestamp.timeIntervalSince(xValue)) <
            abs(entry2.timestamp.timeIntervalSince(xValue))
        }
        
        selectedDataPoint = closestEntry
    }
    
    private func formatXAxisLabel(date: Date) -> String {
        let formatter = DateFormatter()
        
        switch timeframe {
        case .day:
            formatter.dateFormat = "HH:mm"
        case .week:
            formatter.dateFormat = "E"
        case .month:
            formatter.dateFormat = "d"
        case .year:
            formatter.dateFormat = "MMM"
        }
        
        return formatter.string(from: date)
    }
    
    private func formatLifeExpectancy(_ years: Double) -> String {
        let totalYears = Int(years)
        let months = Int((years - Double(totalYears)) * 12)
        
        if months > 0 {
            return "\(totalYears)y \(months)m"
        } else {
            return "\(totalYears)y"
        }
    }
    
    private func formatChange(_ change: Double) -> String {
        let prefix = change > 0 ? "+" : ""
        return "\(prefix)\(String(format: "%.1f", change))y"
    }
    
    private func formatDate(_ date: Date) -> String {
        let formatter = DateFormatter()
        formatter.dateStyle = .short
        formatter.timeStyle = .short
        return formatter.string(from: date)
    }
}

#Preview {
    let sampleData = [
        PredictionHistoryEntry(
            timestamp: Date().addingTimeInterval(-86400 * 7),
            lifeExpectancy: 75.2,
            avgHeartRate: 72,
            dailySteps: 8500,
            sleepHours: 7.5,
            exerciseMinutes: 30,
            contributingFactors: ["Good sleep", "Regular exercise"]
        ),
        PredictionHistoryEntry(
            timestamp: Date().addingTimeInterval(-86400 * 6),
            lifeExpectancy: 75.4,
            avgHeartRate: 70,
            dailySteps: 9200,
            sleepHours: 8.0,
            exerciseMinutes: 45,
            contributingFactors: ["Excellent sleep", "High activity"]
        ),
        PredictionHistoryEntry(
            timestamp: Date().addingTimeInterval(-86400 * 5),
            lifeExpectancy: 75.1,
            avgHeartRate: 75,
            dailySteps: 7800,
            sleepHours: 6.5,
            exerciseMinutes: 20,
            contributingFactors: ["Poor sleep"]
        ),
        PredictionHistoryEntry(
            timestamp: Date().addingTimeInterval(-86400 * 4),
            lifeExpectancy: 75.6,
            avgHeartRate: 68,
            dailySteps: 10500,
            sleepHours: 8.2,
            exerciseMinutes: 60,
            contributingFactors: ["Great workout", "Excellent sleep"]
        ),
        PredictionHistoryEntry(
            timestamp: Date(),
            lifeExpectancy: 75.8,
            avgHeartRate: 69,
            dailySteps: 11000,
            sleepHours: 8.0,
            exerciseMinutes: 50,
            contributingFactors: ["Consistent exercise", "Good sleep"]
        )
    ]
    
    TrendChartView(historicalData: sampleData, timeframe: .week)
}
